/**
 * Tonmly Template Enhancement Script
 * Provides utility functions and enhancements for Tonmly profile templates
 * Foundation for future template features and customizations
 */

(function() {
    'use strict';
    
    // Utility: Get URL parameters
    function getURLParameter(name) {
        const urlParams = new URLSearchParams(window.location.search);
        return urlParams.get(name);
    }
    
    // Utility: Check if element is empty or contains only placeholders
    function isEmpty(element) {
        if (!element) return true;
        
        const content = element.innerHTML.trim();
        if (content === '') return true;
        if (content.includes('{') && content.includes('}')) return true;
        if (content.replace(/<!--.*?-->/g, '').trim() === '') return true;
        
        return false;
    }
    
    // Utility: Clean up placeholder text
    function cleanPlaceholders() {
        const placeholderElements = document.body.querySelectorAll('*:not(script):not(style)');
        placeholderElements.forEach(el => {
            if (el.innerHTML && typeof el.innerHTML === 'string' && el.innerHTML.includes('{') && el.innerHTML.includes('}')) {
                // Remove any unprocessed placeholders
                el.innerHTML = el.innerHTML.replace(/{[^{}]+}/g, '');
                
                // Hide empty elements after placeholder removal
                if (el.innerHTML.trim() === '') {
                    el.style.display = 'none';
                }
            }
        });
    }
    
    // Utility: Detect user's device/platform
    function detectPlatform() {
        const userAgent = navigator.userAgent.toLowerCase();
        return {
            isMobile: /mobile|android|iphone|ipad|tablet/.test(userAgent),
            isIOS: /ipad|iphone|ipod/.test(userAgent) && !window.MSStream,
            isAndroid: /android/.test(userAgent),
            isSafari: /safari/.test(userAgent) && !/chrome/.test(userAgent),
            isChrome: /chrome/.test(userAgent),
            isFirefox: /firefox/.test(userAgent)
        };
    }
    
    // Utility: Smooth scroll to element
    function smoothScrollTo(element, offset = 0) {
        if (element) {
            const elementPosition = element.offsetTop - offset;
            window.scrollTo({
                top: elementPosition,
                behavior: 'smooth'
            });
        }
    }
    
    // Utility: Add click tracking (for analytics)
    function trackClick(elementSelector, eventName) {
        const elements = document.querySelectorAll(elementSelector);
        elements.forEach(element => {
            element.addEventListener('click', function() {
                // Future: Add analytics tracking here
                console.log(`Tracked click: ${eventName}`);
                
                // Example: Google Analytics 4 event (when implemented)
                // gtag('event', eventName, {
                //     'custom_parameter': 'value'
                // });
            });
        });
    }
    
    // Utility: Copy text to clipboard
    function copyToClipboard(text) {
        if (navigator.clipboard) {
            return navigator.clipboard.writeText(text);
        } else {
            // Fallback for older browsers
            const textArea = document.createElement('textarea');
            textArea.value = text;
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            try {
                document.execCommand('copy');
                return Promise.resolve();
            } catch (err) {
                return Promise.reject(err);
            } finally {
                document.body.removeChild(textArea);
            }
        }
    }
    
    // Utility: Add share functionality
    function initializeSharing() {
        const shareButtons = document.querySelectorAll('[data-share]');
        shareButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                const shareType = this.dataset.share;
                const url = encodeURIComponent(window.location.href);
                const title = encodeURIComponent(document.title);
                
                let shareUrl;
                switch(shareType) {
                    case 'twitter':
                        shareUrl = `https://twitter.com/intent/tweet?url=${url}&text=${title}`;
                        break;
                    case 'facebook':
                        shareUrl = `https://www.facebook.com/sharer/sharer.php?u=${url}`;
                        break;
                    case 'linkedin':
                        shareUrl = `https://www.linkedin.com/sharing/share-offsite/?url=${url}`;
                        break;
                    case 'copy':
                        copyToClipboard(window.location.href).then(() => {
                            // Show success message
                            console.log('URL copied to clipboard');
                        });
                        return;
                }
                
                if (shareUrl) {
                    window.open(shareUrl, '_blank', 'width=600,height=400');
                }
            });
        });
    }
    
    // Utility: Enhanced link handling
    function enhanceLinks() {
        const links = document.querySelectorAll('a[href]');
        links.forEach(link => {
            // Add external link indicators
            if (link.hostname !== window.location.hostname) {
                link.setAttribute('target', '_blank');
                link.setAttribute('rel', 'noopener noreferrer');
            }
            
            // Track link clicks
            link.addEventListener('click', function() {
                const linkText = this.textContent.trim();
                const linkUrl = this.href;
                console.log(`Link clicked: ${linkText} -> ${linkUrl}`);
            });
        });
    }
    
    // Main initialization function
    function init() {
        try {
            // Basic utilities that can run on all templates
            cleanPlaceholders();
            enhanceLinks();
            initializeSharing();
            
            // Add platform-specific classes
            const platform = detectPlatform();
            if (platform.isMobile) document.body.classList.add('mobile-device');
            if (platform.isIOS) document.body.classList.add('ios-device');
            if (platform.isAndroid) document.body.classList.add('android-device');
            
            // Future: Add more template enhancements here
            // - Theme switching
            // - Accessibility improvements
            // - Performance optimizations
            // - Custom animations
            // - Form handling
            // - Progressive Web App features
            
        } catch (error) {
            console.error('Template enhancement error:', error);
        }
    }
    
    // Expose utilities globally for template-specific use
    window.XYZTemplateUtils = {
        getURLParameter,
        isEmpty,
        detectPlatform,
        smoothScrollTo,
        trackClick,
        copyToClipboard
    };
    
    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
})();